
#include "judge.h"

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>

#include <avr/io.h>
#include <util/delay.h>

#include "lcd.h"


static activeLCD;

/* HD44780 LCD port connections */
#define HD44780_PORT C

#define HD44780_RS PORT5
#define HD44780_RW PORT4

#define HD44780_D4 PORT0
#define HD44780_D5 PORT1
#define HD44780_D6 PORT2
#define HD44780_D7 PORT3

// enable is on portD
#define HD44780_E1  PORT2
#define HD44780_E2  PORT3
#define HD44780_E3  PORT4


#define GLUE(a, b)     a##b
#define PORT(x)        GLUE(PORT, x)
#define PIN(x)         GLUE(PIN, x)
#define DDR(x)         GLUE(DDR, x)

#define HD44780_E_PORTOUT	PORTD

#define HD44780_PORTOUT    PORT(HD44780_PORT)
#define HD44780_PORTIN     PIN(HD44780_PORT)
#define HD44780_DDR        DDR(HD44780_PORT)

#define HD44780_DATABITS \
(_BV(HD44780_D4)|_BV(HD44780_D5)|_BV(HD44780_D6)|_BV(HD44780_D7))

#define HD44780_BUSYFLAG 0x80

/*
 * Send one pulse to the E signal (enable).  Mind the timing
 * constraints.
 */
static inline void
hd44780_pulse_e(void)
{
	// pulse E depending on which LCD is selected

	if(1==activeLCD)
		HD44780_E_PORTOUT |= _BV(HD44780_E1);
	else if(2==activeLCD)
		HD44780_E_PORTOUT |= _BV(HD44780_E2);
	else
		HD44780_E_PORTOUT |= _BV(HD44780_E3);

	_delay_us(0.4);		/* guarantee 500 ns high */

	if(1==activeLCD)
		HD44780_E_PORTOUT &= ~_BV(HD44780_E1);
	else if(2==activeLCD)
		HD44780_E_PORTOUT &= ~_BV(HD44780_E2);
	else
		HD44780_E_PORTOUT &= ~_BV(HD44780_E3);


}

/*
 * Send one nibble out to the LCD controller.
 */
static void
hd44780_outnibble(uint8_t n, uint8_t rs)
{
  uint8_t x;

  HD44780_PORTOUT &= ~_BV(HD44780_RW);
  if (rs)
    HD44780_PORTOUT |= _BV(HD44780_RS);
  else
    HD44780_PORTOUT &= ~_BV(HD44780_RS);
  x = (HD44780_PORTOUT & ~HD44780_DATABITS) | (n & HD44780_DATABITS);
  HD44780_PORTOUT = x;
  hd44780_pulse_e();
}

/*
 * Send one byte to the LCD controller.  As we are in 4-bit mode, we
 * have to send two nibbles.
 */
void
hd44780_outbyte(uint8_t b, uint8_t rs)
{
  hd44780_outnibble(b >> 4, rs);
  hd44780_outnibble(b & 0xf, rs);
}

/*
 * Read one nibble from the LCD controller.
 */
static uint8_t
hd44780_innibble(uint8_t rs)
{
  uint8_t x;

  HD44780_PORTOUT |= _BV(HD44780_RW);
  HD44780_DDR &= ~HD44780_DATABITS;
  if (rs)
    HD44780_PORTOUT |= _BV(HD44780_RS);
  else
    HD44780_PORTOUT &= ~_BV(HD44780_RS);
  hd44780_pulse_e();
  x = HD44780_PORTIN & HD44780_DATABITS;
  HD44780_DDR |= HD44780_DATABITS;
  HD44780_PORTOUT &= ~_BV(HD44780_RW);

  return x & HD44780_DATABITS;
}

/*
 * Read one byte (i.e. two nibbles) from the LCD controller.
 */
uint8_t
hd44780_inbyte(uint8_t rs)
{
  uint8_t x;

  x = hd44780_innibble(rs) << 4;
  x |= hd44780_innibble(rs);

  return x;
}

/*
 * Wait until the busy flag is cleared.
 */
void
hd44780_wait_ready(void)
{
  while (hd44780_incmd() & HD44780_BUSYFLAG) ;
}

/*
 * Initialize the LCD controller.
 *
 * The initialization sequence has a mandatory timing so the
 * controller can safely recognize the type of interface desired.
 * This is the only area where timed waits are really needed as
 * the busy flag cannot be probed initially.
 */
void
hd44780_init(void)
{

  //HD44780_DDR = _BV(HD44780_RS) | _BV(HD44780_RW) | HD44780_DATABITS;

  _delay_ms(50);		/* 40 ms needed for Vcc = 2.7 V */
  hd44780_outnibble(HD44780_FNSET(1, 0, 0) >> 4, 0);
  _delay_ms(8);
  hd44780_outnibble(HD44780_FNSET(1, 0, 0) >> 4, 0);
  _delay_ms(1);
  hd44780_outnibble(HD44780_FNSET(1, 0, 0) >> 4, 0);

  hd44780_outnibble(HD44780_FNSET(0, 1, 0) >> 4, 0);
  hd44780_wait_ready();
  hd44780_outcmd(HD44780_FNSET(0, 1, 0));
  hd44780_wait_ready();
  hd44780_outcmd(HD44780_DISPCTL(0, 0, 0));
  hd44780_wait_ready();
}

/*
 * Setup the LCD controller.  First, call the hardware initialization
 * function, then adjust the display attributes we want.
 */
void
lcd_init(void)
{

//	activeLCD=1;

  hd44780_init();

  /*
   * Clear the display.
   */
  hd44780_outcmd(HD44780_CLR);
  hd44780_wait_ready();

  /*
   * Entry mode: auto-increment address counter, no display shift in
   * effect.
   */
  hd44780_outcmd(HD44780_ENTMODE(1, 0));
  hd44780_wait_ready();

  /*
   * Enable display, activate non-blinking cursor.
   */
  hd44780_outcmd(HD44780_DISPCTL(1, 0, 0));
  hd44780_wait_ready();
}

/*
 * Send character c to the LCD display.  After a '\n' has been seen,
 * the next character will first clear the display.
 */
int
lcd_putchar(char c, FILE *unused)
{
  static bool nl_seen;
  static bool lf_seen;

  if (nl_seen && c != '\n')
    {
      /*
       * First character after newline, clear display and home cursor.
       */
      hd44780_wait_ready();
      hd44780_outcmd(HD44780_CLR);
      hd44780_wait_ready();
      hd44780_outcmd(HD44780_HOME);
      hd44780_wait_ready();
      hd44780_outcmd(HD44780_DDADDR(0));

      nl_seen = false;
    }

  if (lf_seen && c != '\r')
    {
      /*
       * First character after newline, clear display and home cursor.
       */
     // hd44780_wait_ready();
     // hd44780_outcmd(HD44780_CLR);
      //hd44780_wait_ready();
      //hd44780_outcmd(HD44780_HOME);
      hd44780_wait_ready();
      hd44780_outcmd(HD44780_DDADDR(0x4c));


      lf_seen = false;
    }


  if (c == '\n')
    {
      nl_seen = true;
    }
  else if (c=='\r')
  {
		lf_seen=true;
  }
  else
    {
      hd44780_wait_ready();
      hd44780_outdata(c);
    }

  return 0;
}

void
LCD_puts(char *buf)
{
	if(buf!=0)
	{
		while(*buf)
		{
			lcd_putchar(*buf++,0);
		}
	}
}


void
lcd(U8 lcdnum)
{
	activeLCD=lcdnum;
}
