/*
** imin.c   - iReady Minimum OS.
**
** Created July 1 1998 MWJ iReady Co. 
**
*/
                      
#include <stdio.h>
#include <itypes.h>
#include "imin.h"

static  BOOLEAN     imin_initialized=IFALSE;
static  U16         imin_tick;
static  CALLBACK    *cbhead;
static  CALLBACK    *cbfree;
static  TCALLBACK   *tcbhead;

static	TCALLBACK	*tcbfree;
static  CALLBACK    cbitems[MAX_CALLBACKS];
static  TCALLBACK   tcbitems[MAX_TCALLBACKS];


/*
** imin_init() - initialize the OS
*/
void 
imin_init()
{
U16 i;

    if(imin_initialized==ITRUE)
        return;

    /* initialize the callback structures and mark initialized */
    cbhead=NULL;
    tcbhead=NULL;

    cbfree=&cbitems[0];
    for(i=0;i<MAX_CALLBACKS-2;i++)
        cbitems[i].next=&cbitems[i+1];
    cbitems[i].next=NULL;

    tcbfree=&tcbitems[0];
    for(i=0;i<MAX_TCALLBACKS-2;i++)
        tcbitems[i].next=&tcbitems[i+1];
    tcbitems[i].next=NULL;   
    
    //
    // Initialize Tick and Hardware
    //
    imin_SysInit(); 

    imin_initialized=ITRUE;
}

/*
** imin_init() - initialize the OS
*/
void 
imin_Shtudown()
{	
	imin_initialized=IFALSE;
	imin_Shutdown();	
}


/*
** imin_cbAdd() - add a callback function to be executed as often
**      as possible.
** returns a handle to be used by remove.
** parameters:
**      priority - to be inserted after any "higher" priorit callbacks
**          (the lower the value the higher the priority).
**      function - pointer to function to call
**      call_ptr - a pointer to pass as the argument for this callback
**          (so we can have multiple callbacks using the same function,
**          but using a diferent argument.)
*/
void *imin_cbAdd(priority, function, call_ptr)
U16     priority;
CB_FUNC function;
void    *call_ptr;
{
CALLBACK	*tptr=NULL;
CALLBACK	*lptr;

    /* do nothing if not initialized*/
    if(imin_initialized==ITRUE)
    {
        /* try to get a free calback structure */
        tptr=cbfree;
        if(tptr!=NULL)
        {

            cbfree=cbfree->next;

            /* fill in callback structure */
            tptr->priority=priority;
            tptr->function=function;
            tptr->call_ptr=call_ptr;
            tptr->next=NULL;
            tptr->incall=0;

            /* insert in list */
            if(cbhead==NULL)
            {
                cbhead=tptr;
            }
            else
            {
                if(cbhead->priority>priority)
                {
                    tptr->next=cbhead;
                    cbhead=tptr;
                }
                else
                {
                    lptr=cbhead;
                    while((lptr->next!=NULL) || (lptr->next->priority<priority))
                        lptr=lptr->next;
                    tptr->next=lptr->next;
                    lptr->next=tptr;
                }
            }
        }
    }
    return(tptr);
}

/*
** imin_cbRemove() - removes a previously added callback function.
**      returns a success or fail.
** parameters:
**      handle - a value previously returned by cbAdd.
*/
BOOLEAN
imin_cbRemove(handle)
void    *handle;
{
BOOLEAN      ret=IFALSE;
CALLBACK    *tptr,*lptr;

    tptr=(CALLBACK *)handle;

    if(handle==cbhead)
    {
        cbhead=cbhead->next;
        tptr->next=cbhead;
        cbhead=tptr;
        ret=ITRUE;
    }
    else
    {
        /* scan through the list and try to find us */
        lptr=cbhead;
        while((lptr->next!=NULL) || (lptr->next!=tptr))
            lptr=lptr->next;

        /* if not null, we've found it, remove it */
        if(lptr->next!=NULL)
        {
            /* remove item from the callback list */
            lptr->next=lptr->next->next;
            /* add item back on the free list */
            tptr->next=cbfree;
            cbfree->next=tptr;
            ret=ITRUE;
        }
    }
}

/*
** imin_tcbAdd() - add a callback function to be executed after a
**      certain number of ticks, keep these functions as
**      short in execution time as possible.
**
** returns a handle to be used by remove.
** parameters:
**      period - amount of time in ms to wait between calls.
**      function - pointer to function to call
**      call_ptr - a pointer to pass as the argument for this callback
**          (so we can have multiple callbacks using the same function,
**          but using a diferent argument.)
*/
void *imin_tcbAdd(period, function, call_ptr)
U16     period;
CB_FUNC function;
void    *call_ptr;
{                 
TCALLBACK	*tptr;

    /* do nothing if not initialized*/
    if(imin_initialized==ITRUE)
    {
        /* try to get a free calback structure */
        tptr=tcbfree;
        if(tptr!=NULL)
        {
            /* remove from the free list */
            tcbfree=tcbfree->next;

            /* fill in callback structure */
            tptr->period=period;
            tptr->nexttime=period+imin_ticCnt();
            tptr->function=function;
            tptr->call_ptr=call_ptr;

            /* insert in list */
            tptr->next=tcbhead;
            tcbhead=tptr;
        }
    }
    return((void *)tptr);
}

/*
** imin_tcbRemove() - removes a previously added callback function.
**      returns a success or fail.
** parameters:
**      handle - a value previously returned by cbAdd.
*/
BOOLEAN
imin_tcbRemove(handle)
void    *handle;
{
BOOLEAN      ret=IFALSE;
TCALLBACK	*tptr;
TCALLBACK	*lptr;

    tptr=(TCALLBACK *)handle;

    if(handle==tcbhead)
    {
        tcbhead=tcbhead->next;
        tptr->next=tcbhead;
        tcbhead=tptr;
        ret=ITRUE;
    }
    else
    {
        /* scan through the list and try to find us */
        lptr=tcbhead;
        while((lptr->next!=NULL) || (lptr->next!=tptr))
            lptr=lptr->next;

        /* if not null, we've found it, remove it */
        if(lptr->next!=NULL)
        {
            /* remove item from the callback list */
            lptr->next=lptr->next->next;
            /* add item back on the free list */
            tptr->next=tcbfree;
            tcbfree->next=tptr;
            ret=ITRUE;
        }
    }

}


/*
** imin_callback() - Actual Callback routine that trys to
**      run all the callbacks in order of their priority.
*/
U16 imin_callback()
{       
CALLBACK	*tptr;
U16 ret=0;     


	tptr=cbhead;
	while((!ret) && (tptr!=NULL))
	{
        if(0==tptr->incall)
        {
            tptr->incall=1;
            ret=tptr->function(tptr->call_ptr);
            tptr->incall=0;
        }
        tptr=tptr->next;
	}
	return(ret);
}



/*
** imin_tickcallback() - called by the timer interrupt service routine
**      Interrupt condition must be solved before calling this routine.
*/
void  imin_tickcallback()
{
TCALLBACK   *tptr;
S16         ret;
static U8   sem=0;

	/* incerment tick count */
	imin_tick++;
    return;
	/* check to see if we are exicuting a callback already */
	if(sem==0)
	{
        /* set semiphore */
        sem=1;

        /* enable interrupts */
        ENABLE_INTERRUPTS();

        tptr=tcbhead;
        /* check to see if we need to call any of the callbacks */
        while(tptr != NULL)
        {
            if((tptr->nexttime-imin_ticCnt())<0)
            {
                ret=tptr->function(tptr->call_ptr);
                tptr->nexttime=tptr->period+imin_ticCnt();
                if(ret<0)
                {
                    /* remove timer call back, do not exicute
                       any others this tick */
                    imin_tcbRemove(tptr);
                    break;
                }
            }
            tptr=tptr->next;
        }
        /* clear semiphore */
        sem=0;
    }
}



/*
** imin_ticCnt() - returns the current tick count.
*/
S16 
imin_ticCnt()
{
S16 ret_tick;

    /* disable */
    OS_ENTER_CRITICAL();
    ret_tick=imin_tick;
    /* enable */
    OS_EXIT_CRITICAL();
    return(imin_tick);
}




